function [subMatrix newA] = getSubMatrix(obj,varargin)
%SUBMATRIX Return a submatrix of the given adjacency matrix
%
%   SUBMATRIX = GETSUBMATRIX(obj,cells) where 'obj' is an AdjacencyMatrix
%   object, and 'cells' is a cell array of strings, returns AdjacencyMatrix
%   object only containing the interactions of the cells specified in
%   'cells'
%
%   SUBMATRIX = GETSUBMATRIX(obj,cells,'remove') will return an
%       AdjacencyMatrix object in which the given cells will be removed
%       from obj
%
%   SUBMATRIX = GETSUBMATRIX(obj,preCells,postCells) where 'obj' is an
%       AdjacencyMatrix object, and 'preCells' and 'postCells' are cell
%       arrays of strings, returns AdjacencyMatrix object only containing
%       the interactions of the cells specified in 'preCells' on the
%       pre-synaptic side and 'postCells' on the post-synaptic side
%
%   [SUBMATRIX A] = GETSUBMATRIX(obj, ...) will return both an
%       AdjacencyMatrix object SUBMATRIX and a sparse matrix A which was
%       used to build that adjacency matrix.  The order of rows and columns
%       in A is guaranteed to be the same as that give by the parameters

switch nargin
    case 2
        preCells = varargin{1};
        postCells = varargin{1}';
        
        
    case 3
        if strcmp(varargin{2}, 'remove')
            preCells = setdiff({obj.cells.name}, varargin{1});
            postCells = preCells';
        else
            preCells = varargin{1};
            postCells = varargin{2};
            if isempty(preCells)
                preCells = obj.rowLabels;
            end
            if isempty(postCells)
                postCells = obj.columnLabels;
            end
        end
        
end

% Pre-allocate sparse matrix
newA = sparse(numel(preCells),numel(postCells));
newA2 = sparse(numel(preCells),numel(postCells));

% Loop through non-zero values of obj.A
[I J] = find(obj.A);
for kk = 1:numel(I)
    newi = strcmp(preCells,obj.rowLabels{I(kk)});
    newj = strcmp(postCells,obj.columnLabels{J(kk)});
    newA(newi,newj) = obj.A(I(kk),J(kk));
end



% Could be optimized, probably without using subsref and the double loop
% % for i = 1:numel(preCells)
% %     for j = 1:numel(postCells)
% %         newA2(i,j) = subsref(obj,struct('type','()','subs',{{preCells{i} postCells{j}}}));
% %     end
% % end
% %
% % disp(full(sum(sum(newA-newA2))));

subMatrix = AdjacencyMatrix(newA,preCells,postCells);

end

